/*
 * netprefix.c - Access routines to the NetPrefix ILMI MIB variable
 *
 * Written by Scott W. Shumate
 * 
 * Copyright (c) 1995 Telecommunications & Information
 * Sciences Laboratory, The University of Kansas 
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this
 * software and its documentation is hereby granted,
 * provided that both the copyright notice and this
 * permission notice appear in all copies of the software,
 * derivative works or modified versions, and any portions
 * thereof, that both notices appear in supporting
 * documentation, and that the use of this software is
 * acknowledged in any publications resulting from using
 * the software.
 * 
 * TISL ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION AND DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS
 * SOFTWARE.
 * 
 * Telecommunications & Information Science Lab
 * 2291, Irving Hill Road Lawrence, KS 66046
 * (913)-864-7757
 * http://www.tisl.ukans.edu
 * 
 * The development of this software was generously
 * supported by research grants from Sprint corporation,
 * and we would like to express our thanks.
 */

#include "netprefix.h"
#include "atmd.h"
#include "util.h"

#define COMPONENT "NETPREFIX"
#define NETPREFIX_LEN 12
#define NETPREFIX_STRINGLEN '\15'
#define INDEX_LEN 15
#define LOCALUNI '\0'
#define VALID 1
#define INVALID 0

AsnOid atmNetPrefixStatus = {NETPREFIX_LEN, "\53\06\01\04\01\202\141\02\07\01\01\03"};

static AsnOid atmNetPrefix = {0, NULL};

typedef struct NetPrefixNode
{
  AsnOid *name;
  struct NetPrefixNode *prev;
  struct NetPrefixNode *next;
} NetPrefixNode;

AsnOid *accessNetPrefix(void)
{
  if(atmNetPrefix.octs == NULL)
    return NULL;

  return(&atmNetPrefix);
}

void deleteNetPrefix(void)
{
  NetPrefixNode *prefix, *nextPrefix;

  for(prefix = (NetPrefixNode *) MIBdelete(&atmNetPrefixStatus);
      prefix != NULL;
      prefix = nextPrefix)
    {
      nextPrefix = prefix->next;
      free(prefix->name->octs);
      free(prefix->name);
      free(prefix);
    }

  if(atmNetPrefix.octs != NULL)
    {
      free(atmNetPrefix.octs);
      atmNetPrefix.octs = NULL;
    }
}

AsnInt getNetPrefix(VarBind *varbind, Variable *var)
{
  int cmp;
  NetPrefixNode *prefix;
  AsnOid *varBindName;

  varBindName = &varbind->name;
  if(AsnOidSize(varBindName) != NETPREFIX_LEN + INDEX_LEN ||
     varbind->name.octs[NETPREFIX_LEN] != LOCALUNI ||
     varbind->name.octs[NETPREFIX_LEN + 1] != NETPREFIX_STRINGLEN)
    return NOSUCHNAME;

  for(prefix = (NetPrefixNode *) var->value, cmp = -1;
      prefix != NULL && (cmp = AsnOidCompare(varBindName, prefix->name)) == -1;
      prefix = prefix->next);

  if(cmp != 0)
    return NOSUCHNAME;

  varbind->value = Asn1Alloc(sizeof(struct ObjectSyntax));
  varbind->value->choiceId = OBJECTSYNTAX_SIMPLE;
  varbind->value->a.simple = Asn1Alloc(sizeof(struct SimpleSyntax));
  varbind->value->a.simple->choiceId = SIMPLESYNTAX_NUMBER;
  varbind->value->a.simple->a.number = VALID;
  return NOERROR;
}


AsnInt getnextNetPrefix(VarBind *varbind, Variable *var)
{
  NetPrefixNode *prefix;
  AsnOid *varBindName;

  varBindName = &varbind->name;
  for(prefix = (NetPrefixNode *) var->value;
      prefix != NULL && AsnOidGreater(varBindName, prefix->name);
      prefix = prefix->next);

  if(prefix == NULL)
    return NOSUCHNAME;

  varbind->name.octs = Asn1Alloc(prefix->name->octetLen);
  AsnOidCopy(varBindName, prefix->name);

  varbind->value = Asn1Alloc(sizeof(struct ObjectSyntax));
  varbind->value->choiceId = OBJECTSYNTAX_SIMPLE;
  varbind->value->a.simple = Asn1Alloc(sizeof(struct SimpleSyntax));
  varbind->value->a.simple->choiceId = SIMPLESYNTAX_NUMBER;
  varbind->value->a.simple->a.number = VALID;

  return NOERROR;
}

AsnInt setNetPrefix(VarBind *varbind, Variable *var)
{
  int cmp;
  NetPrefixNode *prefix, *newPrefix;
  AsnOid *varBindName;

  varBindName = &varbind->name;
  if(varbind->value->choiceId != OBJECTSYNTAX_SIMPLE ||
     varbind->value->a.simple->choiceId != SIMPLESYNTAX_NUMBER ||
     (varbind->value->a.simple->a.number != VALID && 
      varbind->value->a.simple->a.number != INVALID))
    return BADVALUE;
      
  if(AsnOidSize(varBindName) != NETPREFIX_LEN + INDEX_LEN ||
     varBindName->octs[NETPREFIX_LEN] != LOCALUNI || 
     varBindName->octs[NETPREFIX_LEN + 1] != NETPREFIX_STRINGLEN)
    return NOSUCHNAME;

  for(prefix = (NetPrefixNode *) var->value, cmp = -1;
      prefix != NULL && (cmp = AsnOidCompare(varBindName, prefix->name)) == -1;
      prefix = prefix->next);

  if(varbind->value->a.simple->a.number == VALID && cmp != 0)
    {
      newPrefix = alloc_t(NetPrefixNode);
      newPrefix->name = alloc_t(AsnOid);
      newPrefix->name->octs = alloc(varBindName->octetLen);
      AsnOidCopy(newPrefix->name, varBindName);
      Q_INSERT_BEFORE((NetPrefixNode *) var->value, newPrefix, prefix);
      if(atmNetPrefix.octs == NULL)
	{
	  atmNetPrefix.octetLen = varBindName->octetLen - NETPREFIX_LEN - 2;
	  atmNetPrefix.octs = alloc(atmNetPrefix.octetLen);
	  memcpy(atmNetPrefix.octs, &varBindName->octs[NETPREFIX_LEN + 2], atmNetPrefix.octetLen);
	}
    }
  else if (varbind->value->a.simple->a.number == INVALID && cmp == 0)
    {
      Q_REMOVE((NetPrefixNode *) var->value, prefix);
    }

  return NOERROR;
}






