/*************************************************************************
  (c) Copyright.  Digital Equipment Corporation, 1995.  All Rights
  Reserved.

  Permission is hereby granted to use, copy, modify, or enhance this 
  software freely, as long as the foregoing copyright of Digital Equipment
  Corporation and this notice are retained on the software.  This 
  software may not be distributed or sublicensed for a fee.  Digital      
  makes this software available "AS IS" and without warranties of any
  kind.  
 *************************************************************************/
/*
 * Marko Kiiskila carnil@cs.tut.fi 
 * 
 * Tampere University of Technology - Telecommunications Laboratory
 *
 * Permission to use, copy, modify and distribute this
 * software and its documentation is hereby granted,
 * provided that both the copyright notice and this
 * permission notice appear in all copies of the software,
 * derivative works or modified versions, and any portions
 * thereof, that both notices appear in supporting
 * documentation, and that the use of this software is
 * acknowledged in any publications resulting from using
 * the software.
 * 
 * TUT ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION AND DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS
 * SOFTWARE.
 * 
 */

/*
* Module Name:
*   system.h
*   
* Abstract:
*   This module defines system specific items.  The definitions in this file
*   are available to all of the system modules.
*
* Authors:
*   TLR - Theodore L. Ross
*
* Modification History:
*   Date        Name   Description 
*   27-Oct-94   TLR    Created.
*   03-Feb-95   PJR    Updated bit masks to allow for 32 bit user_data.
*                      Added USER_DATA_LEH_MASK to extract the header.
*                      Removed len field from INT_BUF, length is passed
*                       separately to the lower driver.
*                      Added USER_DATA_ADD_HDR_MASK to test the add
*                       lan emulation header bit.
*   06-Mar-95   JAD    Removed INT_BUF.
*
* Description:
*
--*/

/* USER_DATA
 *
 *  This value is passed between upper layer SAPs and the lower device
 *  driver through the Connection Manager (CM) with every packet.  For transmit
 *  packets, it contains information about the buffer format and how the buffer
 *  should be deallocated by the driver.
 *
 *    +---+---+---+----------------+-------------------------------------+
 *    | 1 | 1 | 1 |      13        |                  16                 |
 *    +---+---+---+----------------+-------------------------------------+
 *      |   |   |         |                |
 *      |   |   |         |                +-- header_data
 *      |   |   |         +------------------- reserved
 *      |   |   +----------------------------- add_snap_header
 *      |   +--------------------------------- add_le_header
 *      +------------------------------------- internal_source
 *
 *  USER_DATA contains the following fields:
 *
 *    internal_source...If this bit is set, the transmit packet associated
 *                      with this user_data was generated internally.  This
 *                      means that the packet was generated by a component of
 *                      the ATM system (i.e. Signaling, UME, Lan Emulation,
 *                      etc.).  The driver must treat this buffer as an
 *                      internal buffer and not as an operating system
 *                      generated packet.
 *
 *    add_le_header.....If this bit is set, the transmit packet associated
 *                      with this user_data is a LAN Emulation packet.  The
 *                      driver must add the supplied LAN Emulation header to
 *                      the packet as an added fragment.
 *
 *    add_snap_header...If this bit is set, the transmit packet associated
 *                      with this user_data is a Classical-IP (RFC 1577)
 *                      packet.  The driver must prepend a SNAP header with
 *                      the supplied Protocol ID (PID).
 *
 *    reserved..........Reserved for future extension.
 *
 *    header_data.......Either the two octet LAN Emulation header or the PID
 *                      of the SNAP header to be prepended to the packet if
 *                      the 'add_le_header' or 'add_snap_header' bits are set
 *                      respectively.
 *
 *  Invariant:
 *
 *    At most one of the following bits may be set in the user_data:
 *
 *       internal_source
 *       add_le_header
 *       add_snap_header
 *
 *    The lower driver must never be required to prepend a header on an
 *    internally generated packet.
 */
#ifndef SYSTEM_H
#define SYSTEM_H

#define USER_DATA_NORMAL            (0x00000000L)
#define USER_DATA_NORMAL_LE(leh)    (0x40000000L | (leh))
#define USER_DATA_NORMAL_SNAP(pid)  (0x20000000L | (pid))
#define USER_DATA_ADD_HDR_MASK      (0x40000000L)        /* Obsolete */
#define USER_DATA_ADD_LEH_MASK      (0x40000000L)
#define USER_DATA_ADD_SNAP_MASK     (0x20000000L)
#define USER_DATA_INTERNAL          (0x80000000L)
#define USER_DATA_LEH_MASK          (0x0000FFFFL)
#define USER_DATA_PID_MASK          (0x0000FFFFL)

/* ATM_LINK_STATUS
 *
 *   This type enumerates the possible link states that are visible to
 *   registered SAP clients.
 *
 *   LINK_DOWN......The physical link is not available.  This occurs
 *                  when signal synchronization with the network peer is
 *                  lost.
 *
 *   LINK_PHY_UP....The physical link has become available.  This
 *                  signifies that the ATM port has detected the presence
 *                  of a peer on the other end of the wire/fiber.
 *
 *   LINK_LINE_UP...The Line-Up protocol has completed successfully.
 *                  This indication allows protocols like SSCOP to begin
 *                  bringing themselves to life.
 *
 *   LINK_SIG_UP....SSCOP has initialized and the signaling stack is ready
 *                  to accept SVC requests.
 *
 */

typedef enum
   {
   LINK_DOWN,
   LINK_PHY_UP,
   LINK_LINE_UP,
   LINK_SIG_UP
   } ATM_LINK_STATE;

/* ATM_LINK_EVENT
 *
 *   This type enumerates the possible link state events that can be
 *   reported to the Connection Manager.  These events may result in
 *   changes to the link state.
 *
 *   LINK_EVENT_PHY_DOWN........Carrier was lost on the physical port.
 *                              The link state will transition to
 *                              LINK_DOWN after a hysteresis period
 *                              elapses (around 3 seconds).
 *
 *   LINK_EVENT_PHY_UP..........Carrier is up on the physical port.
 *                              Link state transitions immediately from
 *                              LINK_DOWN to LINK_PHY_UP.
 *
 *   LINK_EVENT_LINE_UP.........The Line-Up exchange with the network
 *                              peer is complete.  Link state transitions
 *                              immediately from LINK_PHY_UP to LINK_LINE_UP.
 *
 *   LINK_EVENT_SIG_UP..........The Signaling transport (i.e. QSAAL) has
 *                              completed its connection with its peer.
 *                              The link state transitions immediately from
 *                              LINK_LINE_UP to LINK_SIG_UP.
 *
 *   LINK_EVENT_SIG_DOWN........The Signaling transport has lost contact
 *                              with its peer.  The link state transitions
 *                              immediately from LINK_SIG_UP to LINK_LINE_UP.
 *                              All SVCs are deleted in an orderly way (i.e.
 *                              the driver is asked to teardown the VCs).
 *
 *   LINK_EVENT_RESET_START.....The driver is resetting.  The link state
 *                              transitions immediately from wherever it is
 *                              to LINK_DOWN.  All SVCs are deleted harshly
 *                              (i.e. the records are simply deleted, no
 *                              notification goes to either the signaling
 *                              function or the device driver).
 *
 *   LINK_EVENT_RESET_COMPLETE..Driver reset is complete, link state stays
 *                              at LINK_DOWN.  All PVCs are re-opened on
 *                              the driver.
 */
typedef enum
   {
   LINK_EVENT_PHY_DOWN,
   LINK_EVENT_PHY_UP,
   LINK_EVENT_LINE_UP,
   LINK_EVENT_SIG_UP,
   LINK_EVENT_SIG_DOWN,
   LINK_EVENT_RESET_START,
   LINK_EVENT_RESET_COMPLETE
   } ATM_LINK_EVENT;

#endif /* SYSTEM_H */
