/* qlib.c - run-time library */
 
/* Written 1995,1996 by Werner Almesberger, EPFL-LRC */
 

int q_dump = 0;


#ifdef DUMP_MODE

#include <stdarg.h>
#include <stdio.h>
#include <unistd.h>

#include "common.h"
#include "qlib.h"
#include "op.h"


static int debug = 0;


void q_report(int severity,const char *msg,...)
{
    va_list ap;

    if (!debug) return;
    va_start(ap,msg);
    vprintf(msg,ap);
    printf("\n");
    va_end(ap);
    if (severity == Q_FATAL) exit(1);
}

#endif


static int q_test(unsigned char *table,int pos)
{
    return !!(table[pos >> 3] & (1 << (pos & 7)));
}


static void q_set(unsigned char *table,int pos)
{
    table[pos >> 3] |= 1 << (pos & 7);
}


/* slightly ugly */


static void q_put(unsigned char *table,int pos,int size,unsigned long value)
{
    int end;
 
    q_report(Q_DEBUG,"put %d %d %ld",pos,size,value);
    end = pos+size;
    if (((pos | size) & 7) && ((pos ^ (end-1)) & ~7))
	q_report(Q_FATAL,"unsupported alignment");
    if (size <= 8) {
	unsigned char *here;
	int shift;

	here = &table[pos >> 3];
	shift = pos & 7;
	*here = (*here & ~(((1 << size)-1) << shift)) | value << shift;
    }
    else {
	table = table+end/8-1;
	while (size > 0) {
	    *table-- = value;
	    value >>= 8;
	    size -= 8;
        }
    }
}


static unsigned long q_get(unsigned char *table,int pos,int size)
{
    unsigned long value;
    int end;
 
    q_report(Q_DEBUG,"get %d %d ...",pos,size);
    end = pos+size;
    if (((pos | size) & 7) && ((pos ^ (end-1)) & ~7))
	q_report(Q_FATAL,"unsupported alignment");
    if (size <= 8) value = (table[pos >> 3] >> (pos & 7)) & ((1 << size)-1);
    else {
	table += pos >> 3;
	value = 0;
	while (size > 0) {
	    value = (value << 8) | *table++;
	    size -= 8;
        }
    }
    q_report(Q_DEBUG,"  %ld",value);
    return value;
}


#if 0
static void q_put(unsigned char *table,int pos,int size,unsigned long value)
{
    unsigned char *byte;
 
    byte = &table[pos >>3];
    if (pos & 7) {
	*byte++ |= value << (pos & 7);
	value >>= 8-(pos & 7);
	size -= 8-(pos & 7);
    }
    while (size > 0) {
	size -= 8;
	*byte++ |= value;
	value >>= 8;
    }
}
#endif
 

static void q_copy(unsigned char *src,int pos,unsigned char *dst,int size)
{
    src += pos >> 3;
    pos &= 7;
    if (pos+size <= 8) *dst |= *src & (((1 << size)-1) << pos);
    else {
	if (pos) {
	    *dst++ |= *src++ & (0xff << pos);
	    size -= 8-pos;
	}
	while (size >= 8) {
	    *dst++ = *src++;
	    size -= 8;
	}
	if (size > 0) *dst |= *src & ((1 << size)-1);
    }
}


void q_start(void)
{
    q_init_global();
}


static int q_init(Q_DSC *dsc)
{
    size_t bytes;

    dsc->data = malloc((size_t) Q_DATA_BYTES);
    if (!dsc->data) {
	perror("out of memory");
	return -1;
    }
    memcpy(dsc->data,q_initial,Q_DATA_BYTES);
    bytes = (Q_FIELDS+7) >> 3;
    dsc->required = malloc(bytes);
    if (!dsc->required) {
	perror("out of memory");
	return -1;
    }
    memset(dsc->required,0,bytes);
    dsc->field_present = malloc(bytes);
    if (!dsc->field_present) {
	perror("out of memory");
	return -1;
    }
    memset(dsc->field_present,0,bytes);
    bytes = (Q_GROUPS+7) >> 3;
    dsc->group_present = malloc(bytes);
    if (!dsc->group_present) {
	perror("out of memory");
	return -1;
    }
    memset(dsc->group_present,0,bytes);
    if (!Q_VARLEN_FIELDS) dsc->length = NULL;
    else {
	dsc->length = malloc(sizeof(int)*Q_VARLEN_FIELDS);
	if (!dsc->length) {
	    perror("out of memory");
	    return -1;
	}
    }
    dsc->error = 0;
    return 0;
}


static void use_group(Q_DSC *dsc,int group)
{
    int *scan;

    while (group != -1) {
	q_set(dsc->group_present,group);
	for (scan = groups[group].required; scan && *scan != -1; scan++)
	    q_set(dsc->required,*scan);
	group = groups[group].parent;
    }
}


void q_assign(Q_DSC *dsc,int field,unsigned long value)
{
    int *walk;

    if (field < 0 || field >= Q_FIELDS)
	q_report(Q_FATAL,"invalid field value (%d)",field);
    if (!fields[field].values) {
	q_set(dsc->field_present,field);
	q_put(dsc->data,fields[field].pos,fields[field].size,value);
	use_group(dsc,fields[field].parent);
    }
    else {
	if (q_test(dsc->field_present,field))
	    q_report(Q_FATAL,"can't change field %d",field);
	q_set(dsc->field_present,field);
	q_put(dsc->data,fields[field].pos,fields[field].size,value);
	for (walk = fields[field].values; walk[1] != -1; walk += 2)
	    if (*walk == value || *walk == -2) {
		use_group(dsc,walk[1]);
		return;
	    }
	q_report(Q_ERROR,"invalid value (%d in field %d)",value,field);
	dsc->error = 1;
    }
}


void q_write(Q_DSC *dsc,int field,const void *buf,int size)
{
    if (field < 0 || field >= Q_FIELDS)
	q_report(Q_FATAL,"invalid field value (%d)",field);
    if (fields[field].pos & 7)
	q_report(Q_FATAL,"invalid use of q_write (%d)",field);
    if (fields[field].actual >= 0) {
	if (size > fields[field].size/8) {
	    q_report(Q_ERROR,"%d bytes too big for %d byte field %d",size,
	      fields[field].size/8,field);
	    dsc->error = 1;
	    return;
	}
	dsc->length[fields[field].actual] = size;
    }
    else if ((fields[field].pos | fields[field].size) & 7)
	    q_report(Q_FATAL,"field %d is neither var-len nor well-shaped",
	      field);
    memcpy(dsc->data+(fields[field].pos/8),buf,(size_t) size);
    q_set(dsc->field_present,field);
    use_group(dsc,fields[field].parent);
}


int q_present(const Q_DSC *dsc,int field)
{
    if (field < 0) {
	if (field < -Q_GROUPS)
	    q_report(Q_FATAL,"invalid group number (%d)",field);
	return q_test(dsc->group_present,-field-1);
    }
    else {
	if (field >= Q_FIELDS)
	    q_report(Q_FATAL,"invalid field number (%d)",field);
	if (q_test(dsc->field_present,field)) return 1;
	return q_test(dsc->group_present,fields[field].parent);
    }
}


unsigned long q_fetch(const Q_DSC *dsc,int field)
{
    if (field < 0 || field >= Q_FIELDS)
	q_report(Q_FATAL,"invalid field value (%d)",field);
    return q_get(dsc->data,fields[field].pos,fields[field].size);
}


int q_length(const Q_DSC *dsc,int field)
{
    if (field < 0 || field >= Q_FIELDS)
	q_report(Q_FATAL,"invalid field value (%d)",field);
    if (fields[field].pos & 7)
	q_report(Q_FATAL,"invalid use of q_length (%d)",field);
    if (fields[field].actual < 0)
	q_report(Q_FATAL,"field %d is not var-len",field);
    return dsc->length[fields[field].actual];
}


int q_read(Q_DSC *dsc,int field,void *buf,int size)
{
    int len;

    if (field < 0 || field >= Q_FIELDS)
	q_report(Q_FATAL,"invalid field value (%d)",field);
    if (fields[field].pos & 7)
	q_report(Q_FATAL,"invalid use of q_read (%d)",field);
    if (fields[field].actual >= 0) len = dsc->length[fields[field].actual];
    else if (!(fields[field].size & 7)) len = fields[field].size >> 3;
	else q_report(Q_FATAL,"field %d is not byte-sized (%d bits)",field,
	      fields[field].size);
    if (size < len) {
	q_report(Q_ERROR,"%d bytes too big for %d byte buffer (field %d)",len,
	  size,field);
	dsc->error = 1;
	return -1;
    }
    memcpy(buf,dsc->data+(fields[field].pos/8),len);
    return len;
}


#define LENGTH_STACK 10


typedef struct {
    int pos,size;
    unsigned char *start;
} LEN_BUF;


static int q_compose(Q_DSC *dsc,unsigned char *buf,int size)
{
    LEN_BUF stack[LENGTH_STACK];
    unsigned char *pos;
    int *pc;
    int j,i,sp;

    for (i = 0; i < Q_FIELDS; i++)
	if (q_test(dsc->required,i) && !q_test(dsc->field_present,i))
	    q_report(Q_ERROR,"required field %d is missing",i);
    memset(buf,0,(size_t) size);
    if (q_dump)
	for (j = 0; j < 100; j += 20) {
	    fprintf(stderr,"%3d:",j);
	    for (i = 0; i < 20; i++) fprintf(stderr," %02X",dsc->data[i+j]);
	    putc('\n',stderr);
	}
    pos = buf;
    pc = construct; 
    sp = 0;
    while (1) {
	q_report(Q_DEBUG,"%d(%d):",pc-construct,pos-buf);
	switch (*pc++) {
	    case OP_COPY:
		if (size < *pc) {
		    q_report(Q_ERROR,"not enough space (%d < %d)",size,*pc);
		    dsc->error = 1;
		    return -1;
		}
		q_report(Q_DEBUG,"copy %d %d %d",pc[1],pos-buf,pc[2]);
		q_copy(dsc->data,pc[1],pos,pc[2]);
		if (q_dump) {
		    for (i = 0; i < 50; i++) fprintf(stderr,"%02X ",buf[i]);
		    putc('\n',stderr);
		}
		pos += *pc;
		size -= *pc;
		pc += 3;
		break;
	    case OP_COPYVAR:
		if (size < dsc->length[*pc]) {
		    q_report(Q_ERROR,"not enough space (%d < %d)",size,
		      dsc->length[*pc]);
		    dsc->error = 1;
		    return -1;
		}
		memcpy(pos,dsc->data+pc[1]/8,dsc->length[*pc]);
		pos += dsc->length[*pc];
		size -= dsc->length[*pc];
		pc += 3;
		break;
	    case OP_BEGIN_LEN:
		if (size < *pc) {
		    q_report(Q_ERROR,"not enough space (%d < %d)",size,*pc);
		    dsc->error = 1;
		    return -1;
		}
		if (sp == LENGTH_STACK) {
		    q_report(Q_ERROR,"length stack overflow");
		    dsc->error = 1;
		    return -1;
		}
		stack[sp].pos = pc[1]; /* not used */
		stack[sp].size = pc[2];
		stack[sp].start = pos;
		pos += *pc; /* allocate length here */
		size -= *pc;
		sp++;
		pc += 3;
		break;
	    case OP_END_LEN:
		if (!sp--) q_report(Q_FATAL,"length stack underflow");
		q_put(stack[sp].start,0,stack[sp].size,
		  (size_t) ((pos-stack[sp].start)-((stack[sp].size+7) >> 3)));
		break;
	    case OP_IFGROUP:
		if (q_test(dsc->group_present,*pc++)) pc++;
		else pc += *pc+1;
		break;
#if 0
	    case OP_CASE:
		{
		    int len;

		    for (len = *pc++; len; len--)
			if (!q_test(dsc->group_present,*pc++)) pc++;
			else {
			    pc += *pc+1;
			    break;
			}
		    if (!len)
			q_report(Q_FATAL,"multi failed (pc %d)",pc-construct);
		}
		break;
#endif
	    case OP_JUMP:
		pc += *pc+1;
		break;
	    case OP_END:
		return pos-buf;
	    default:
		q_report(Q_FATAL,"unrecognized opcode %d",pc[-1]);
		return -1; /* for gcc */
	}
    }
}


static int _q_parse(Q_DSC *dsc,unsigned char *buf,int size)
{
    unsigned char *stack[LENGTH_STACK];
    unsigned char *pos,*end;
    int *pc;
    int i,sp;

    end = buf+size;
    pos = buf;
    pc = parse; 
    sp = 0;
    while (1) {
	q_report(Q_DEBUG,"%d(%d):",pc-parse,pos-buf);
	switch (*pc++) {
#ifdef DUMP_MODE
	    case OP_DUMP:
		{
		    unsigned long value;
		    int len;

		    for (i = dump_fields[*pc].level; i; i--) printf("  ");
		    printf("%s =",dump_fields[*pc++].name);
		    len = *pc == OP_COPYVAR ? (end-pos)*8 : pc[3];
		    if (len <= 32) {
			const SYM_NAME *sym;

			value = q_get(pos,pc[2] & 7,pc[3]);
			if (!(sym = dump_fields[pc[-1]].sym))
			    printf(" %ld (0x%lx)\n",value,value);
			else {
			    while (sym->name)
				if (sym->value == value) break;
				else sym++;
			    if (sym->name) printf(" %s\n",sym->name);
			    else printf(" %ld (0x%lx)\n",value,value);
			}
		    }
		    else {
			for (i = 0; i < len/8; i++)
			    printf(" %02x",pos[pc[1]/8+i]);
			putchar('\n');
		    }
		}
		break;
#endif
	    case OP_COPY:
		if (pos+*pc > end) {
		    q_report(Q_ERROR,"not enough space (%d+%d > %d)",pos-buf,
		      *pc,end-buf);
		    dsc->error = 1;
		    return -1;
		}
		q_report(Q_DEBUG,"copy %d %d %d",pc[1],pos-buf,pc[2]);
		q_copy(pos,pc[1] & 7,dsc->data+(pc[1] >> 3),pc[2]);
		if (q_dump) {
		    for (i = 0; i < 20; i++)
			fprintf(stderr,"%02X ",dsc->data[i]);
		    putc('\n',stderr);
		}
		pos += *pc;
		pc += 3;
		break;
	    case OP_COPYVAR:
		{
		    int len;

		    len = end-pos;
		    if (len > pc[2]) len = pc[2];
		    memcpy(dsc->data+pc[1]/8,pos,(size_t) len);
		    q_report(Q_DEBUG,"len %d for %d",len,*pc);
		    dsc->length[*pc] = len;
		    pos += len;
		    pc += 3;
		    break;
		}
	    case OP_BEGIN_LEN:
		if (pos+*pc > end) {
		    q_report(Q_ERROR,"not enough space (%d+%d > %d)",pos-buf,
		      *pc,end-buf);
		    dsc->error = 1;
		    return -1;
		}
		if (sp == LENGTH_STACK) {
		    q_report(Q_ERROR,"length stack overflow");
		    dsc->error = 1;
		    return -1;
		}
		stack[sp] = end;
		end = pos+q_get(pos,pc[1] & 7,pc[2])+*pc;
		if (end > stack[sp]) q_report(Q_FATAL,"length has grown");
		pos += *pc;
		sp++;
		pc += 3;
		break;
	    case OP_END_LEN:
		if (!sp--) q_report(Q_FATAL,"length stack underflow");
		end = stack[sp];
		break;
	    case OP_CASE:
		{
		    int len,value,group;

		    if (pos+*pc > end) {
			q_report(Q_ERROR,"not enough space (%d+%d > %d)",
			  pos-buf,*pc,end-buf);
			dsc->error = 1;
			return -1;
		    }
		    value = q_get(pos,pc[1] & 7,pc[2]);
		    pos += *pc;
		    pc += 3;
		    for (len = *pc++; len; len--)
			if (*pc != value && *pc != -1) pc += 3;
			else {
			    pc++;
			    for (group = *pc++; group != -1; group =
			      groups[group].parent)
				q_set(dsc->group_present,group);
			    pc += *pc+1;
			    break;
			}
		    if (!len) {
			q_report(Q_ERROR,"case failed (pc %d)",pc-construct);
			return -1;
		    }
		}
		break;
	    case OP_JUMP:
		pc += *pc+1;
		break;
	    case OP_IFEND:
		q_report(Q_DEBUG,"ifend - %d/%d",pos-buf,end-buf);
		if (pos == end) pc += *pc;
		pc++;
		break;
	    case OP_END:
		return 0;
	    default:
		q_report(Q_FATAL,"unrecognized opcode %d",pc[-1]);
		return -1; /* for gcc */
	}
    }
}


static int q_parse(Q_DSC *dsc,unsigned char *buf,int size)
{
    int error;
    int j,i;
    int *p;

    if (q_dump)
	for (j = 0; j < 100; j += 20) {
	    fprintf(stderr,"%3d:",j);
	    for (i = 0; i < 20; i++) fprintf(stderr," %02X",dsc->data[i+j]);
	    putc('\n',stderr);
	}
    error = _q_parse(dsc,buf,size);
    if (error) return error;
    if (q_dump) {
	putc('\n',stderr);
	for (j = 0; j < 100; j += 20) {
	    fprintf(stderr,"%3d:",j);
	    for (i = 0; i < 20; i++) fprintf(stderr," %02X",dsc->data[i+j]);
	    putc('\n',stderr);
	}
	for (i = 0; i < Q_GROUPS; i++)
	    if (q_test(dsc->group_present,i))
		for (p = groups[i].required; p && *p != -1; p++)
		    fprintf(stderr,"%d: %ld / 0x%lx\n",*p,
		      q_get(dsc->data,fields[*p].pos,fields[*p].size),
		      q_get(dsc->data,fields[*p].pos,fields[*p].size));
    }
    return 0;
}


int q_open(Q_DSC *dsc,void *buf,int size)
{
    dsc->buffer = NULL;
    q_init(dsc);
    return q_parse(dsc,buf,size);
}


int q_create(Q_DSC *dsc,void *buf,int size)
{
   dsc->buffer = buf;
   dsc->buf_size = size;
   q_init(dsc);
   return 0;
}


int q_close(Q_DSC *dsc)
{
    int size;

    if (dsc->buffer && !dsc->error)
	size = q_compose(dsc,dsc->buffer,dsc->buf_size);
    free(dsc->data);
    free(dsc->required);
    free(dsc->field_present);
    free(dsc->group_present);
    if (dsc->length) free(dsc->length);
    return dsc->error ? -1 : dsc->buffer ? size : 0;
}


#ifdef DUMP_MODE

int main(int argc,const char **argv)
{
    unsigned char msg[5000]; /* should be large enough for that */
    Q_DSC dsc;
    int len,c;

    debug = argc != 1;
    len = 0;
    while (scanf("%x",&c) == 1) msg[len++] = c;
    q_open(&dsc,msg,len);
    return 0;
}

#endif
