/*
 * Copyright (c) 2007, intarsys consulting GmbH
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 *
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * - Neither the name of intarsys nor the names of its contributors may be used
 *   to endorse or promote products derived from this software without specific
 *   prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package de.intarsys.tools.functor;

import java.util.HashSet;
import java.util.Set;

/**
 * An {@link IArgs} implementation allowing to concat two {@link IArgs}
 * together. Lookup will be performed in "args" first. If nothing is found,
 * lookup is done in "fallbackArgs".
 * 
 */
public class ChainedArgs implements IArgs {

	private IArgs args;

	private IArgs fallbackArgs;

	/**
	 * Create new {@link ChainedArgs} where <code>args</code> are always
	 * looked up first. If lookup fails, <code>fallbackArgs</code> are used.
	 * 
	 * @param args
	 *            The main {@link IArgs} to use for lookup
	 * @param fallbackArgs
	 *            The fallback {@link IArgs} for lookup
	 */
	public ChainedArgs(IArgs args, IArgs fallbackArgs) {
		super();
		this.args = args;
		this.fallbackArgs = fallbackArgs;
	}

	public void add(Object object) {
		args.add(object);
	}

	public void clear() {
		args.clear();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.intarsys.tools.functor.IArgs#get(int)
	 */
	public Object get(int index) {
		Object result = args.get(index);
		if (result == null) {
			return fallbackArgs.get(index);
		} else if (result instanceof IArgs) {
			Object fallbackResult = fallbackArgs.get(index);
			if (fallbackResult instanceof IArgs) {
				return new ChainedArgs((IArgs) result, (IArgs) fallbackResult);
			}
		}
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.intarsys.tools.functor.IArgs#get(int, java.lang.Object)
	 */
	public Object get(int index, Object defaultValue) {
		Object result = args.get(index);
		if (result == null) {
			return fallbackArgs.get(index, defaultValue);
		} else if (result instanceof IArgs) {
			Object fallbackResult = fallbackArgs.get(index);
			if (fallbackResult instanceof IArgs) {
				return new ChainedArgs((IArgs) result, (IArgs) fallbackResult);
			}
		}
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.intarsys.tools.functor.IArgs#get(java.lang.String)
	 */
	public Object get(String name) {
		Object result = args.get(name);
		if (result == null) {
			if (!args.isDefined(name)) {
				result = fallbackArgs.get(name);
			}
		} else if (result instanceof IArgs) {
			Object fallbackResult = fallbackArgs.get(name);
			if (fallbackResult instanceof IArgs) {
				result = new ChainedArgs((IArgs) result, (IArgs) fallbackResult);
			}
		}
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.intarsys.tools.functor.IArgs#get(java.lang.String,
	 *      java.lang.Object)
	 */
	public Object get(String name, Object defaultValue) {
		Object result = args.get(name);
		if (result == null) {
			if (!args.isDefined(name)) {
				result = fallbackArgs.get(name, defaultValue);
			}
		} else if (result instanceof IArgs) {
			Object fallbackResult = fallbackArgs.get(name);
			if (fallbackResult instanceof IArgs) {
				result = new ChainedArgs((IArgs) result, (IArgs) fallbackResult);
			}
		}
		return result;
	}

	/**
	 * The main (primary) {@link IArgs}.
	 * 
	 * @return The main (primary) {@link IArgs}.
	 */
	public IArgs getArgs() {
		return args;
	}

	/**
	 * The fallback (secondary) {@link IArgs};
	 * 
	 * @return The fallback (secondary) {@link IArgs};
	 */
	public IArgs getFallbackArgs() {
		return fallbackArgs;
	}

	public boolean isDefined(int index) {
		if (args.isDefined(index)) {
			return true;
		}
		return fallbackArgs.isDefined(index);
	}

	public boolean isDefined(String name) {
		if (args.isDefined(name)) {
			return true;
		}
		return fallbackArgs.isDefined(name);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.intarsys.tools.functor.IArgs#isIndexed()
	 */
	public boolean isIndexed() {
		return args.isIndexed() || fallbackArgs.isIndexed();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.intarsys.tools.functor.IArgs#isNamed()
	 */
	public boolean isNamed() {
		return args.isNamed() || fallbackArgs.isNamed();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.intarsys.tools.functor.IArgs#names()
	 */
	public Set names() {
		Set names = new HashSet(args.names());
		names.addAll(fallbackArgs.names());
		return names;
	}

	public void put(int index, Object value) {
		args.put(index, value);
	}

	public void put(String name, Object value) {
		args.put(name, value);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.intarsys.tools.functor.IArgs#size()
	 */
	public int size() {
		if (isIndexed()) {
			return Math.max(args.size(), fallbackArgs.size());
		}
		return names().size();
	}

	@Override
	public String toString() {
		return ArgTools.toString(this, "");
	}

}
